#!/bin/bash
#
# spam_ip.sh
# Simple-minded IP address extractor from email header
# Created: September 2019
# Authors: Issakar Doude and Wirawan Purwanto
#
# This script is a part of DeapSECURE hands-on activities
#
# How does it work?
# -----------------
#
# Script to extract an IP address from the last "Received:" field in
# an email address, then performs a lookup query from a database
# to yield the country which has that IP address.
# This will be the suspected originating country of the email.
#
# Usage:
#
#     spam_ip.sh <DIRECTORY>
#
# where DIRECTORY is a directory containing email files to process.
#
# CAVEAT: This is a very simplistic script!
#
# * All files in DIRECTORY will be opened and parsed
#
# * It does not always get the correct source IP address of the sender.
#
# To give learners opportunity to understand how this script works,
# we try to adhere to only syntax and commands we have in the lesson.
# Exceptions:
#
# * shell command substitution via `$( )`
# * math operation via `$(( ))`
#
# We try to make this script runnable under the following shells:
# bash, dash, ksh, zsh.
# (Tested 2020-09-30 on a MacOS laptop.)

EMAIL_DIR=$1

#Loops over email files in the directory given as parameter
for EMAIL in "$EMAIL_DIR"/*
do 
    # Extract last 'Received' IP from the current email being processed
    # 1. Filter all the "Received:" lines
    # 2. Then extract the IP number quartet
    # 3. Then select only the very last one of those quartets
    IP=$(cat "$EMAIL" \
           | grep "^Received:" \
           | grep -E -o "(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)" \
           | tail -n 1)

    if [ -z "$IP" ]; then
        echo "${EMAIL}||Fail to get source IP|"
    else
        # Convert the quad to a 32-bit int to look up the country.
        # Because there are only four numbers, we will not use a `for` loop here
        # (that is ok too).
        NUM1=$(echo "$IP" | cut -d . -f 1)
        NUM2=$(echo "$IP" | cut -d . -f 2)
        NUM3=$(echo "$IP" | cut -d . -f 3)
        NUM4=$(echo "$IP" | cut -d . -f 4)

        # Construct the 32-bit number of the IP address
        # We use the math operator
        # (NOTE: The operators like * will not be expanded to "a list of all files"
        # like the usual UNIX command)
        IP_INT=$(( $NUM1 * 16777216  +  $NUM2 * 65536  +  $NUM3 * 256  +  $NUM4 ))

        # Slightly more sophisticated:
        #IP_INT=0
        #for n in $(echo $IP | tr "." " ")
        #do
        #    IP_INT=$(( $IP_INT * 256 + $n ))
        #done

        #Prepare the DB query string
        QUERY="SELECT country_code, country FROM IP_Country WHERE (min_ip <= $IP_INT) AND ($IP_INT <= max_ip);"
        #echo "$QUERY"

        #Get the country code and name from the database
        CC2_COUNTRY=$(echo "$QUERY" | sqlite3 -separator '|' IP2LOCATION-LITE-DB1.sqlite)

        # Print the result in one line
        # (tabular format, fields separated by "|"
        echo "${EMAIL}|${IP}|${CC2_COUNTRY}"
    fi
done

